#ifndef BSTREAM_H
#define BSTREAM_H

#include "memutil.h"
#include "shift.h"

typedef struct {
	int *data;
	int bufpos;
	int pos;
	int buf;
} bstream_t;

#define BSTREAM_TELL(s) ((s).bufpos)

#define BSTREAM_INIT_READ(s, data_arg, pos_arg) do { \
	s.data = (data_arg); \
	s.bufpos = (pos_arg); \
	s.pos = ALIGN_DOWN(s.bufpos + 32, 32); \
	s.buf = swap_bytes(s.data[s.bufpos >> 5]) << (32 - (s.pos - s.bufpos)); \
} while (0)

#define BSTREAM_READ_BIT(s, dest) do { \
	if (s.bufpos == s.pos) { \
		s.buf = swap_bytes(s.data[s.pos >> 5]); \
		s.pos += 32; \
	} \
	SRL(dest, s.buf, 31); \
	s.buf <<= 1; \
	s.bufpos++; \
} while (0)

#define BSTREAM_READ(s, dest, width) do { \
	int _spare = s.pos - s.bufpos; \
	int _w = width; \
	int _over = _w - _spare; \
	SRLV(dest, s.buf, 32 - _w); \
	s.bufpos += _w; \
	if (_over > 0) { \
		_w = _over; \
		s.buf = swap_bytes(s.data[s.pos >> 5]); \
		s.pos += 32; \
		int _part; \
		SRLV(_part, s.buf, 32 - _w); \
		dest |= _part; \
	} \
	if (_w < 32) \
		s.buf <<= _w; \
} while (0)

#define BSTREAM_INIT_WRITE(s, data_arg, pos_arg) do { \
	s.data = (data_arg); \
	s.pos = s.bufpos = (pos_arg); \
} while (0)

static int __inline__ bstream_flush(bstream_t s) {
	int n_bits = s.bufpos - s.pos;
	if (n_bits == 32) {
		s.data[s.pos >> 5] = swap_bytes(s.buf);
	}
	else if (n_bits > 0) {
		int shift_bits = 32 - n_bits - (s.pos & 31);
		int mask = swap_bytes((1 << n_bits) - 1 << shift_bits);
		int old = s.data[s.pos >> 5] & mask;
		int new = swap_bytes(s.buf << shift_bits) & mask;
		int incr = new - old;
		psm(incr, s.data[s.pos >> 5]);
	}
	return s.bufpos;
}

#define BSTREAM_FLUSH(s) s.pos = bstream_flush(s)
#define BSTREAM_SEEK(s, pos) do { \
	BSTREAM_FLUSH(s); \
	BSTREAM_INIT_READ(s, s.data, pos); \
} while (0)

#define BSTREAM_WRITE_BIT(s, value) do { \
	s.buf = s.buf << 1 | (value); \
	s.bufpos++; \
	if ((s.bufpos & 31) == 0) \
		BSTREAM_FLUSH(s); \
} while (0)

#define BSTREAM_WRITE(s, value, width) do { \
	int _spare = 32 - (s.bufpos & 31); \
	int _w = width; \
	unsigned _v = value; \
	int _over = _w - _spare; \
	int _flush; \
	if (_over > 0) { \
		s.buf = s.buf << _spare | _v >> _over; \
		s.bufpos += _spare; \
		_flush = 1; \
	} \
	else if (_w == 32) { \
		s.buf = _v; \
		s.bufpos += _w; \
		_flush = 1; \
	} \
	else { \
		s.buf = s.buf << _w | _v; \
		s.bufpos += _w; \
		_flush = ((s.bufpos & 31) == 0); \
	} \
	if (_flush) \
		BSTREAM_FLUSH(s); \
	if (_over > 0) { \
		s.buf = _v; \
		s.bufpos += _over; \
	} \
} while (0)

#endif
